<?php
/**
 * id-image-upload.php copyright 2025 by The Finest Of Lines Tattoo Co. 17080 Superior Road St. Robert MO, 65584.
 * Main tel# 557-227-2575
 * Art desk tel# 557-227-2576
 * emails: Admin@finestoflines.com, oldbones@finestoflines.net, nikitaraze@gmail.com, bones@finestoflines.net
 * Code blocks by Grok 4 file version: id-image-upload.php-v5
 * Concept & Tuck-pointing by Raydog.
 * 2025-11-02 11:00
 */

require_once __DIR__ . '/vendor/autoload.php';

use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

$manager = new ImageManager(new Driver());

/* ------------------------------------------------------------------ */
/* CORS – allow localhost:8100 + any origin                           */
/* ------------------------------------------------------------------ */
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Credentials: true');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

/* ------------------------------------------------------------------ */
/* PDO connection (once)                                              */
/* ------------------------------------------------------------------ */
$dsn      = 'mysql:host=localhost;dbname=lineup_2026_test;charset=utf8mb4';
$db_user  = getenv('CITADEL_2025_DB_USER') ?: 'data';
$db_pass  = getenv('CITADEL_2025_DB_PASSWORD') ?: 'ncc1701D';

try {
    $pdo = new PDO($dsn, $db_user, $db_pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_bin"
    ]);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'DB connection failed']);
    exit;
}

/* ------------------------------------------------------------------ */
/* Main logic                                                         */
/* ------------------------------------------------------------------ */
$input     = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$lineup_id = $input['lineup_id'] ?? $_POST['lineup_id'] ?? null;
$upload_id = $input['upload_id'] ?? uniqid('id_', true);

if (!$lineup_id) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'lineup_id required']);
    exit;
}

apcu_store("upload_progress_{$upload_id}", 0, 300);

$images    = [];
$total     = 0;
$processed = 0;

/* ------------------- Safe multipart handling ----------------------- */
$uploadedFiles = [];
if (isset($_FILES['files']) && is_array($_FILES['files'])) {
    if (isset($_FILES['files']['tmp_name']) && !is_array($_FILES['files']['tmp_name'])) {
        if ($_FILES['files']['error'] === UPLOAD_ERR_OK && is_uploaded_file($_FILES['files']['tmp_name'])) {
            $uploadedFiles[] = [
                'name'     => $_FILES['files']['name'],
                'tmp_name' => $_FILES['files']['tmp_name'],
                'type'     => $_FILES['files']['type'],
                'error'    => $_FILES['files']['error']
            ];
        }
    } else {
        foreach ($_FILES['files']['tmp_name'] as $k => $tmp) {
            if ($_FILES['files']['error'][$k] === UPLOAD_ERR_OK && is_uploaded_file($tmp)) {
                $uploadedFiles[] = [
                    'name'     => $_FILES['files']['name'][$k],
                    'tmp_name' => $tmp,
                    'type'     => $_FILES['files']['type'][$k],
                    'error'    => $_FILES['files']['error'][$k]
                ];
            }
        }
    }
}

/* ------------------- Process each file --------------------------- */
foreach ($uploadedFiles as $file) {
    $total++;

    /* ---- NEW: DELETE any previous ID images for this lineup_id ---- */
    try {
        $delStmt = $pdo->prepare("DELETE FROM `patron_required_image_b64` WHERE `lineup_id` = ?");
        $delStmt->execute([$lineup_id]);
        error_log("Deleted " . $delStmt->rowCount() . " old ID image(s) for lineup_id {$lineup_id}");
    } catch (Throwable $e) {
        error_log("Failed to delete old ID images: " . $e->getMessage());
        // continue – not fatal
    }

    $result = processIdFile($pdo, $lineup_id, $file, $manager);
    if ($result) $images[] = $result;

    $processed++;
    apcu_store("upload_progress_{$upload_id}", round(($processed / $total) * 100));
}

/* ------------------- Final response ------------------------------ */
$final_progress = $processed === $total ? 100 : (apcu_fetch("upload_progress_{$upload_id}") ?: 0);
echo json_encode([
    'success'     => true,
    'upload_id'   => $upload_id,
    'images'      => $images,
    'progress'    => $final_progress
]);

/* ------------------------------------------------------------------ */
/* Helper – full ID image (WebP) with data-uri prefix                 */
/* ------------------------------------------------------------------ */
function processIdFile(PDO $pdo, int $lineup_id, array $file, ImageManager $manager): ?array
{
    try {
        $img = $manager->read($file['tmp_name']);

        // Full ID image → WebP 80%
        $idImage = $img->toWebp(80);
        $id_b64  = base64_encode($idImage);

        // Full data-uri for DB storage
        $fullDataUri = "data:image/webp;base64,{$id_b64}";

        $stmt = $pdo->prepare(
            "INSERT INTO `patron_required_image_b64` 
             (`image`, `lineup_id`, `mimetype`) 
             VALUES (?, ?, 'image/webp')"
        );
        $stmt->execute([$fullDataUri, $lineup_id]);
        $id = $pdo->lastInsertId();

        return [
            'id'       => $id,
            'base64'   => $id_b64,          // raw base64 for Angular preview
            'filename' => $file['name']
        ];
    } catch (Throwable $e) {
        error_log('ID image error: ' . $e->getMessage());
        return null;
    }
}
?>