<?php
/**
 * api-utils.php  copyright 2025 by The Finest Of Lines Tattoo Co.  17080 Superior Road St. Robert MO, 65584.  
 * Main tel# 557-227-2575
 * Art desk tel# 557-227-2576
 * emails:  Admin@finestoflines.com, oldbones@finestoflines.net, nikitaraze@gmail.com, bones@finestoflines.net
 * Code blocks by Grok 4 file version: api-utils.php-v84
 * Concept & Tuck-pointing by Raydog. 
 * November 13, 2025 17:30
 */

require_once 'vendor/autoload.php';
use Tqdev\PhpCrudApi\Config\Config;
use Tqdev\PhpCrudApi\Database\GenericDB;
require_once 'vendor/tecnickcom/tcpdf/tcpdf.php';

$ENABLE_TESTGEN = true;

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: *');
header('Access-Control-Allow-Methods: OPTIONS, GET, POST');
header('Access-Control-Allow-Credentials: true');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$config = new Config([
    'driver'    => 'mysql',
    'address'   => 'daystrom.finestoflines.net',
    'port'      => '3306',
    'username'  => 'data',
    'password'  => 'ncc1701D',
    'database'  => 'lineup_2026_test',
    'debug'     => false
]);

$db = new GenericDB(
    $config->getDriver(),
    $config->getAddress(),
    $config->getPort(),
    $config->getDatabase(),
    $config->getCommand(),
    $config->getTables(),
    $config->getMapping(),
    $config->getUsername(),
    $config->getPassword(),
    $config->getGeometrySrid()
);

$pdo = $db->pdo();

$action   = $_REQUEST['action'] ?? '';
$lineupId = $_REQUEST['lid'] ?? $_REQUEST['lineup_id'] ?? '';
$debugInfo = [];

/* ------------------------------------------------------------------ */
/*  Helper – strip data:url prefix                                    */
/* ------------------------------------------------------------------ */
function stripDataUrlPrefix($data) {
    if (preg_match('#^data:image/[^;]+;base64,#', $data)) {
        return preg_replace('#^data:image/[^;]+;base64,#', '', $data);
    }
    return $data;
}

/* ------------------------------------------------------------------ */
/*  Helper – resize & fit image to max mm (returns temp jpg path)     */
/* ------------------------------------------------------------------ */
function resizeAndFit($source, $maxWmm, $maxHmm) {
    global $debugInfo;
    $debugInfo[] = "resizeAndFit called – source: $source, max: {$maxWmm}x{$maxHmm}mm";

    if (!file_exists($source)) {
        $debugInfo[] = "Image resize: File not found - $source";
        return null;
    }
    list($w, $h, $type) = @getimagesize($source);
    if (!$w || !$h) {
        $debugInfo[] = "Image resize: Invalid image - $source";
        return null;
    }
    $debugInfo[] = "Original image: {$w}x{$h}px, type: $type";

    $maxWpx = $maxWmm * 11.811;
    $maxHpx = $maxHmm * 11.811;

    $largerDim = max($w, $h);
    $largerMax = ($w > $h) ? $maxWpx : $maxHpx;
    $scale = $largerMax / $largerDim;

    $newW = round($w * $scale);
    $newH = round($h * $scale);

    $dst = imagecreatetruecolor($newW, $newH);
    $src = null;
    switch ($type) {
        case IMAGETYPE_JPEG: $src = imagecreatefromjpeg($source); break;
        case IMAGETYPE_PNG:  $src = imagecreatefrompng($source); break;
        case IMAGETYPE_WEBP:
            if (function_exists('imagecreatefromwebp')) $src = imagecreatefromwebp($source);
            break;
    }
    if (!$src) {
        $debugInfo[] = "Image resize: Unsupported format - $source";
        @unlink($source);
        return null;
    }

    imagecopyresampled($dst, $src, 0, 0, 0, 0, $newW, $newH, $w, $h);
    $tmp = tempnam(sys_get_temp_dir(), 'img_') . '.jpg';
    imagejpeg($dst, $tmp, 92);
    imagedestroy($src); imagedestroy($dst);
    @unlink($source);

    $debugInfo[] = "Image resized: $source → $tmp ($newW×$newH px)";
    return $tmp;
}

/* ------------------------------------------------------------------ */
/*  PDF Generation – shared between testgen & congen                  */
/* ------------------------------------------------------------------ */
function generateContractPdf($pdo, $lineupId, $contractId, $debugInfo, $includeDebug = false) {
    $debugInfo[] = "generateContractPdf called – lineup_id=$lineupId, contract_id=$contractId";

    // ------------------------------------------------------------------
    // 1. Load lineup
    // ------------------------------------------------------------------
    $stmt = $pdo->prepare("SELECT * FROM lineup WHERE lineup_id = ? AND hidden = '0'");
    $stmt->execute([$lineupId]);
    $lineup = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$lineup) throw new Exception('Lineup not found');

    // ------------------------------------------------------------------
    // 2. Load patron (if linked)
    // ------------------------------------------------------------------
    $patron = [];
    if ($lineup['patron_id'] && $lineup['patron_id'] !== '0') {
        $stmt = $pdo->prepare("SELECT * FROM patron_information WHERE patron_id = ?");
        $stmt->execute([$lineup['patron_id']]);
        $patron = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    // ------------------------------------------------------------------
    // 3. Load artists
    // ------------------------------------------------------------------
    $tArtist = $cArtist = [];
    $supervisorLicense = null;
    $isApprentice = false;

    if ($lineup['t_artist_id']) {
        $stmt = $pdo->prepare("SELECT *, status, license FROM artists WHERE artist_id = ?");
        $stmt->execute([$lineup['t_artist_id']]);
        $tArtist = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];

        if ($tArtist && $tArtist['status'] === 'A') {
            $isApprentice = true;
            $supervisorId = $tArtist['license'];
            if ($supervisorId && is_numeric($supervisorId)) {
                $stmt = $pdo->prepare("SELECT license FROM artists WHERE artist_id = ?");
                $stmt->execute([$supervisorId]);
                $sup = $stmt->fetch(PDO::FETCH_ASSOC);
                $supervisorLicense = $sup['license'] ?? '—';
            }
        }
    }

    if ($lineup['c_artist_id']) {
        $stmt = $pdo->prepare("SELECT * FROM artists WHERE artist_id = ?");
        $stmt->execute([$lineup['c_artist_id']]);
        $cArtist = $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    // ------------------------------------------------------------------
    // 4. Prepare placeholders
    // ------------------------------------------------------------------
    $tatId = $contractId;
    $patronFullName = htmlspecialchars(($patron['patron_fname'] ?? '') . ' ' . ($patron['patron_lname'] ?? ''));
    $patronDob = htmlspecialchars($patron['patron_dob'] ?? '');
    $patronAddress = htmlspecialchars($patron['patron_st_address'] ?? '');
    $patronCityStateZip = htmlspecialchars(($patron['patron_city'] ?? '') . ', ' . ($patron['patron_state'] ?? '') . ' ' . ($patron['patron_zip'] ?? ''));
    $patronPhone = htmlspecialchars($patron['patron_phone'] ?? '');
    $price = number_format((float)$lineup['price'], 2);
    $tArtistNickname = htmlspecialchars($tArtist['artist_nickname'] ?? '—');
    $cArtistNickname = htmlspecialchars($cArtist['artist_nickname'] ?? '—');
    $artistLicenseDisplay = $isApprentice ? 'APPRENTICE' : ($tArtist['license'] ?? '—');
    $description = nl2br(htmlspecialchars($lineup['description'] ?? ''));

    $qrData = $lineupId . ' ' . $tatId . ' ' . ($lineup['t_artist_id'] ?? '—') . ' ' . ($lineup['c_artist_id'] ?? '—') . ' ' . ($lineup['patron_id'] ?? '—');

    // ------------------------------------------------------------------
    // 5. Load images (ID + primary design)
    // ------------------------------------------------------------------
    $idImagePath = $designImagePath = null;

    if ($lineup['si_img_id'] && $lineup['si_img_id'] !== '0') {
        $stmt = $pdo->prepare("SELECT image FROM patron_required_image_b64 WHERE si_image_id = ?");
        $stmt->execute([$lineup['si_img_id']]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row && $row['image']) {
            $cleanData = stripDataUrlPrefix($row['image']);
            $tmp = tempnam(sys_get_temp_dir(), 'id_');
            file_put_contents($tmp, base64_decode($cleanData));
            $idImagePath = resizeAndFit($tmp, 63.5, 63.5);
        }
    }

    if ($lineup['primary_design_img'] && $lineup['primary_design_img'] !== '0') {
        $stmt = $pdo->prepare("SELECT design_image FROM lineup_design_images_b64 WHERE design_image_id = ?");
        $stmt->execute([$lineup['primary_design_img']]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row && $row['design_image']) {
            $cleanData = stripDataUrlPrefix($row['design_image']);
            $tmp = tempnam(sys_get_temp_dir(), 'design_');
            file_put_contents($tmp, base64_decode($cleanData));
            $designImagePath = resizeAndFit($tmp, 76.2, 76.2);
        }
    }

    // ------------------------------------------------------------------
    // 6. Load waivers & aftercare
    // ------------------------------------------------------------------
    $waiverTypes = $isApprentice ? "'YN','IN','AP'" : "'YN','IN'";
    $stmt = $pdo->prepare("SELECT waiver_type, waiver_text FROM waiver_text WHERE `use` != '0' AND waiver_type IN ($waiverTypes) ORDER BY FIELD(waiver_type, 'YN', 'IN', 'AP')");
    $stmt->execute();
    $waivers = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt = $pdo->prepare("SELECT aftercare_group, aftercare_text FROM aftercare_text WHERE aftercare_group != '' ORDER BY group_order ASC, waiver_id ASC");
    $stmt->execute();
    $aftercareGroups = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $group = $row['aftercare_group'] ?: 'basic';
        $aftercareGroups[$group][] = $row['aftercare_text'];
    }

    // ------------------------------------------------------------------
    // 7. Load HTML template
    // ------------------------------------------------------------------
    $html = file_get_contents(__DIR__ . '/contract-page1.html');
    if (!$html) throw new Exception('Template missing');

    $html = str_replace([
        '{{tat_id}}', '{{t_artist_nickname}}', '{{c_artist_nickname}}',
        '{{patron_fullname}}', '{{patron_dob}}', '{{patron_address}}',
        '{{patron_city_state_zip}}', '{{patron_phone}}', '${{price}}',
        '{{description}}'
    ], [
        $tatId, $tArtistNickname, $cArtistNickname,
        $patronFullName, $patronDob, $patronAddress,
        $patronCityStateZip, $patronPhone, $price,
        $description
    ], $html);

    // ------------------------------------------------------------------
    // 8. PDF class (shared)
    // ------------------------------------------------------------------
    class ContractPDF extends TCPDF {
        public $tArtist, $cArtist, $tatId;
        public $idImagePath, $designImagePath;
        public $qrData;
        public $waivers = [];
        public $artistLicenseDisplay;
        public $supervisorLicense;
        public $isApprentice;
        public $aftercareGroups = [];
        public $debugInfo = [];

        public function Header() {
            $this->Image(__DIR__ . '/headerbanner.png', 0, 0, 215.9, 25.4);
            $this->SetFont('helvetica', 'B', 13);
            $this->SetTextColor(0,0,0);
            $this->SetXY(105, 3);
            $this->MultiCell(100, 0, "Finest Of Lines Tattoo Company\nSt. Robert, MO\nMain: 557.227.2575  Art Desk: 557.227.2576", 0, 'R');
            $this->SetFillColor(255,255,255);
            $this->SetTextColor(0,0,0);
            $this->SetFont('helvetica', '', 9);
            $this->Rect(0, 21, 215.9, 4.4, 'F');
            $this->SetY(22);
            $this->Cell(71.97, 0, 'Artist: ' . $this->tArtist, 0, 0, 'C');
            $this->Cell(71.97, 0, 'Concept: ' . $this->cArtist, 0, 0, 'C');
            $this->Cell(71.96, 0, 'Consent form #: ' . $this->tatId, 0, 0, 'C');
        }

        public function Footer() {
            $this->SetY(-19);
            $this->SetFillColor(248,248,248);
            $this->Rect(0, $this->GetY(), 215.9, 19, 'F');
            $this->SetTextColor(0,0,0);
            $this->SetFont('helvetica', '', 9);

            $this->SetX(10);
            $this->Cell(0, 0, 'Finest Of Lines Tattoo Co. Est. 2012', 0, 0, 'L');
            $this->Ln(4);
            $this->SetX(10);
            $this->Cell(0, 0, 'Thank you for choosing us to do your tattoo!', 0, 0, 'L');

            $style = ['border' => false, 'padding' => 0, 'fgcolor' => [0,0,0], 'bgcolor' => false];
            $this->write2DBarcode($this->qrData, 'QRCODE,M', 188, $this->GetY() - 4, 18, 18, $style, 'N');
        }

        public function addImages() {
            if ($this->idImagePath && file_exists($this->idImagePath)) {
                list($w, $h) = getimagesize($this->idImagePath);
                $scale = min(63.5 / $w * 25.4, 63.5 / $h * 25.4);
                $newW = $w * $scale / 25.4;
                $newH = $h * $scale / 25.4;
                $offsetX = (63.5 - $newW) / 2;
                $offsetY = (63.5 - $newH) / 2;
                $this->Image($this->idImagePath, 121.1 + $offsetX, 30 + $offsetY, $newW, $newH, '', '', '', false, 300);
            }

            if ($this->designImagePath && file_exists($this->designImagePath)) {
                list($w, $h) = getimagesize($this->designImagePath);
                $scale = min(76.2 / $w * 25.4, 76.2 / $h * 25.4);
                $newW = $w * $scale / 25.4;
                $newH = $h * $scale / 25.4;
                $offsetX = (76.2 - $newW) / 2;
                $offsetY = (76.2 - $newH) / 2;
                $this->Image($this->designImagePath, 121.1 + $offsetX, 96.9 + $offsetY, $newW, $newH, '', '', '', false, 300);
            }
        }

        public function addWaiverPage() {
            $this->AddPage();
            $this->SetMargins(12.7, 30, 12.7);
            $this->SetAutoPageBreak(true, 19);
            $this->SetFont('helvetica', '', 10);
            $this->Ln(10);

            $currentType = null;
            foreach ($this->waivers as $waiver) {
                $type = strtoupper($waiver['waiver_type']);
                $text = htmlspecialchars_decode($waiver['waiver_text']);
                $leftCell = ($type === 'YN') ? 'Yes___ No___' : 'Initial______';

                if ($currentType === 'YN' && $type === 'IN') $this->Ln(3);
                if ($currentType === 'IN' && $type === 'AP') $this->Ln(3);

                $this->MultiCell(25.4, 0, $leftCell, 0, 'L', false, 0);
                $this->MultiCell(0, 0, $text, 0, 'L', false, 1);
                $this->Ln(3);
                $currentType = $type;
            }

            $this->Ln(9);
            $today = date('m/d/Y');
            $lineHeight = 3.5;
            $rowSpacing = 3;

            $this->MultiCell(38.1, $lineHeight, 'Client signature:', 0, 'R', false, 0);
            $this->MultiCell(108.8, $lineHeight, '', 'B', 'L', false, 0);
            $this->MultiCell(38.1, $lineHeight, "Date: {$today}", 0, 'L', false, 1);
            $this->Ln($rowSpacing + $lineHeight);

            $this->MultiCell(38.1, $lineHeight, 'Artist signature:', 0, 'R', false, 0);
            $this->MultiCell(108.8, $lineHeight, '', 'B', 'L', false, 0);
            $this->MultiCell(38.1, $lineHeight, "LIC#: {$this->artistLicenseDisplay}", 0, 'L', false, 1);
            $this->Ln($rowSpacing + $lineHeight);

            if ($this->isApprentice) {
                $this->MultiCell(38.1, $lineHeight, 'Supervising artist:', 0, 'R', false, 0);
                $this->MultiCell(108.8, $lineHeight, '', 'B', 'L', false, 0);
                $this->MultiCell(38.1, $lineHeight, "LIC#: {$this->supervisorLicense}", 0, 'L', false, 1);
            }
        }

        public function addAftercarePages() {
            foreach ($this->aftercareGroups as $group => $items) {
                $this->AddPage();
                $this->SetMargins(12.7, 30, 12.7);
                $this->SetAutoPageBreak(true, 19);
                $this->SetFont('helvetica', 'B', 18);
                $this->Cell(0, 12, strtoupper($group) . ' AFTERCARE', 0, 1, 'C');
                $this->Ln(6);
                $this->SetFont('helvetica', '', 14);
                foreach ($items as $text) {
                    $cleanText = htmlspecialchars_decode($text);
                    $this->writeHTMLCell(0, 0, '', '', '<p>' . $cleanText . '</p>', 0, 1, false, true, 'J');
                    $this->Ln(3);
                }
            }
        }

        public function addDebugPage() {
            $this->AddPage();
            $this->SetFont('helvetica', 'B', 16);
            $this->Cell(0, 10, 'DEBUG REPORT - CONTRACT', 0, 1, 'C');
            $this->Ln(10);
            $this->SetFont('helvetica', '', 11);
            foreach ($this->debugInfo as $line) {
                $this->MultiCell(0, 8, $line, 0, 'L');
            }
            $this->Ln(10);
            $this->SetFont('helvetica', 'B', 12);
            $this->Cell(0, 10, 'Lineup ID: ' . $this->tatId, 0, 1);
            $this->Cell(0, 10, 'Generated: ' . date('Y-m-d H:i:s'), 0, 1);
        }
    }

    // ------------------------------------------------------------------
    // 9. Build PDF
    // ------------------------------------------------------------------
    $pdf = new ContractPDF('P', 'mm', 'LETTER', true, 'UTF-8', false);
    $pdf->SetHeaderMargin(0);
    $pdf->SetFooterMargin(0);
    $pdf->SetLeftMargin(0);
    $pdf->SetRightMargin(0);
    $pdf->SetTopMargin(0);
    $pdf->SetMargins(0, 30, 0);
    $pdf->SetAutoPageBreak(false, 19);

    $pdf->tArtist = $tArtistNickname;
    $pdf->cArtist = $cArtistNickname;
    $pdf->tatId = $tatId;
    $pdf->idImagePath = $idImagePath;
    $pdf->designImagePath = $designImagePath;
    $pdf->qrData = $qrData;
    $pdf->waivers = $waivers;
    $pdf->artistLicenseDisplay = $artistLicenseDisplay;
    $pdf->supervisorLicense = $supervisorLicense;
    $pdf->isApprentice = $isApprentice;
    $pdf->aftercareGroups = $aftercareGroups;
    $pdf->debugInfo = $debugInfo;

    $pdf->AddPage();
    $pdf->writeHTML($html, true, false, true, false, '');
    $pdf->addImages();
    $pdf->addWaiverPage();
    $pdf->addAftercarePages();
    if ($includeDebug) $pdf->addDebugPage();

    $pdfData = $pdf->Output('', 'S');
    $pdfBase64 = base64_encode($pdfData);

    // ------------------------------------------------------------------
    // 10. Save base64 to tattoos_con_gen – ONLY FOR REAL CONTRACTS
    // ------------------------------------------------------------------
    if (is_numeric($tatId)) {
        $save = $pdo->prepare('UPDATE tattoos_con_gen SET contract_data_url = ? WHERE tat_id = ?');
        $save->execute([$pdfBase64, $tatId]);
    }

    return $pdfData;
}

/* ------------------------------------------------------------------ */
/*  TESTGEN – NO AUTH REQUIRED (debug page ON)                        */
/* ------------------------------------------------------------------ */
if ($action === 'testgen' && $ENABLE_TESTGEN) {
    if (!$lineupId || !ctype_digit($lineupId)) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing or invalid lid']);
        exit;
    }

    try {
        $tatId = 'TEST-' . $lineupId;
        $pdfData = generateContractPdf($pdo, $lineupId, $tatId, $debugInfo, true);
        header('Content-Type: application/pdf');
        header('Content-Disposition: inline; filename="test-contract-' . $lineupId . '.pdf"');
        echo $pdfData;
        exit;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed', 'message' => $e->getMessage()]);
        exit;
    }
}

/* ------------------------------------------------------------------ */
/*  AUTH – ONLY REQUIRED FOR congen                                   */
/* ------------------------------------------------------------------ */
if ($action === 'congen') {
    $authKey = $_GET['authkey'] ?? '';
    if (!$authKey) {
        echo json_encode(['success' => false, 'error' => 'Missing authkey']);
        exit;
    }
    $stmt = $pdo->prepare('SELECT user_id FROM authkeys WHERE authkey = ? AND expires > NOW()');
    $stmt->execute([$authKey]);
    if (!$stmt->fetch()) {
        echo json_encode(['success' => false, 'error' => 'Invalid or expired authkey']);
        exit;
    }

    $lineupId = $_GET['lid'] ?? '';
    $contractId = $_GET['cid'] ?? '';

    if (!$lineupId || !$contractId) {
        echo json_encode(['success' => false, 'error' => 'Missing lineup_id or contract_id']);
        exit;
    }

    // --------------------------------------------------------------
    // Validate that the contract_id belongs to this lineup
    // --------------------------------------------------------------
    $stmt = $pdo->prepare("SELECT tat_id, contract_data_url FROM tattoos_con_gen WHERE tat_id = ? AND lineup_id = ?");
    $stmt->execute([$contractId, $lineupId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        echo json_encode(['success' => false, 'error' => 'Contract ID does not match lineup']);
        exit;
    }

    try {
        // --------------------------------------------------------------
        // 1. Check for existing PDF
        // --------------------------------------------------------------
        if (!empty($row['contract_data_url'])) {
            $debugInfo[] = "Cached PDF found for tat_id=$contractId";
            $pdfData = base64_decode($row['contract_data_url']);
            header('Content-Type: application/pdf');
            header('Content-Disposition: inline; filename="contract-' . $contractId . '.pdf"');
            echo $pdfData;
            exit;
        }

        // --------------------------------------------------------------
        // 2. Generate & save new PDF
        // --------------------------------------------------------------
        $debugInfo[] = "No cached PDF – generating new for tat_id=$contractId";
        $pdfData = generateContractPdf($pdo, $lineupId, $contractId, $debugInfo, false);

        header('Content-Type: application/pdf');
        header('Content-Disposition: inline; filename="contract-' . $contractId . '.pdf"');
        echo $pdfData;
        exit;

    } catch (Exception $e) {
        http_response_code(500);
        // Return HTML error page for iframe
        echo "<!DOCTYPE html><html><body style='font-family:Arial;text-align:center;margin-top:50px;'>
              <h2>Contract Generation Failed</h2>
              <p>" . htmlspecialchars($e->getMessage()) . "</p>
              <button onclick='location.reload()'>Retry</button>
              </body></html>";
        exit;
    }
}

/* ------------------------------------------------------------------ */
/*  Fallback                                                          */
/* ------------------------------------------------------------------ */
echo json_encode(['success' => false, 'error' => 'Unknown action']);
?>